<?php

namespace App\Http\Controllers\Landlord\Admin;

use App\Helpers\FlashMsg;
use App\Helpers\Payment\DatabaseUpdateAndMailSend\LandlordPricePlanAndTenantCreate;
use App\Helpers\ResponseMessage;
use App\Http\Controllers\Controller;
use App\Mail\BasicDynamicTemplateMail;
use App\Mail\BasicMail;
use App\Mail\OrderReply;
use App\Mail\PlaceOrder;
use App\Models\CustomDomain;
use App\Models\FormBuilder;
use App\Models\Language;
use App\Models\PackageHistory;
use App\Models\PaymentLogHistory;
use App\Models\PaymentLogs;
use App\Models\PricePlan;
use App\Models\Tenant;
use App\Models\User;
use Barryvdh\DomPDF\Facade as PDF;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Mail;
use Modules\EmailTemplate\Traits\EmailTemplate\Landlord\SubscriptionEmailTemplate;
use Monolog\Handler\IFTTTHandler;

class OrderManageController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:package-order-all-order|package-order-pending-order|package-order-progress-order|
        package-order-edit|package-order-delete|package-order-complete-order|package-order-success-order-page|package-order-order-page-manage
        package-order-order-report');
    }

    private const ROOT_PATH = 'landlord.admin.package-order-manage.';

    public function all_orders(){
        $all_orders = PaymentLogs::latest()->get();
        return view(self::ROOT_PATH.'order-manage-all')->with(['all_orders' => $all_orders]);
    }

    public function view_order($tenant_id)
    {
        abort_if(empty($tenant_id), 404);

        $payment_log = PaymentLogs::where('tenant_id',$tenant_id)->latest()->first();

        if($payment_log->payment_status != 'complete'){
            $payment_history = PaymentLogs::where('tenant_id',$tenant_id)->paginate(5);
        }else{
            $payment_history = PaymentLogHistory::where('tenant_id',$tenant_id)->paginate(5);
        }

        $domain_name = $tenant_id.'.'.env('CENTRAL_DOMAIN');
        return view(self::ROOT_PATH.'order-view',compact('payment_history','domain_name'));
    }

    public function pending_orders(){
        $all_orders = PaymentLogs::where('status','pending')->get();
        return view(self::ROOT_PATH.'order-manage-pending')->with(['all_orders' => $all_orders]);
    }

    public function completed_orders(){
        $all_orders = PaymentLogs::where('status','complete')->orderBy('id','desc')->get();
        return view(self::ROOT_PATH.'order-manage-completed')->with(['all_orders' => $all_orders]);
    }
    public function in_progress_orders(){
        $all_orders = PaymentLogs::where('status','in_progress')->get();
        return view(self::ROOT_PATH.'order-manage-in-progress')->with(['all_orders' => $all_orders]);
    }



    public function order_reminder(Request $request){
        $order_details = PaymentLogs::find($request->id);
        $route = route('landlord.frontend.plan.order',optional($order_details->package)->id);
        //send order reminder mail
        $data['subject'] = __('your order is still in pending at').' '. get_static_option('site_'.get_default_language().'_title');
        $data['message'] = __('hello').' '.$order_details->name .'<br>';
        $data['message'] .= __('your order').' #'.$order_details->id.' ';
        $data['message'] .= __('is still in pending, to complete your order go to');
        $data['message'] .= ' <br> <br><a href="'.$route.'">'.__('go to payment ').'</a>';
        //send mail while order status change

        try {
            Mail::to($order_details->email)->send(new BasicMail($data['message'], $data['subject'] ));
        }catch(\Exception $e){
            return redirect()->back()->with(['type'=> 'danger', 'msg' => $e->getMessage()]);
        }

        return redirect()->back()->with(['msg' => __('PaymentLogs Reminder Mail Send Success...'),'type' => 'success']);
    }

    public function order_delete(Request $request,$id){

        $log = PaymentLogs::findOrFail($id);
        $user = \App\Models\User::findOrFail($log->user_id);

        if(!empty($user)){
            return redirect()->back()->with(['msg' => __('You cannot delete this item, This data is associated with a user, please delete the user then it will be deleted automatically..!'),'type' => 'danger']);
        }
        $log->delete();

        return redirect()->back()->with(['msg' => __('PaymentLogs Deleted Success...'),'type' => 'danger']);
    }


    public function send_mail(Request $request){
        $this->validate($request,[
            'email' => 'required|string|max:191',
            'name' => 'required|string|max:191',
            'subject' => 'required|string',
            'message' => 'required|string',
        ]);
        $subject = str_replace('{site}',get_static_option('site_'.get_default_language().'_title'),$request->subject);
        $data = [
            'name' => $request->name,
            'message' => $request->message,
        ];
        try{
          Mail::to($request->email)->send(new OrderReply($data,$subject));
        }catch(\Exception $e){
            return redirect()->back()->with(['type'=> 'danger', 'msg' => $e->getMessage()]);
        }
        return redirect()->back()->with(['msg' => __('Order Reply Mail Send Success...'),'type' => 'success']);
    }


    public function all_payment_logs(){
        $paymeng_logs = PaymentLogs::all();
        return view('landlord.admin.payment-logs.payment-logs-all')->with(['payment_logs' => $paymeng_logs]);
    }

    public function payment_logs_delete(Request $request,$id){
        PaymentLogs::find($id)->delete();
        return redirect()->back()->with(['msg' => __('Payment Log Delete Success...'),'type' => 'danger']);
    }

    public function change_status(Request $request){

        $this->validate($request,[
            'order_status' => 'required|string|max:191',
            'order_id' => 'required|string|max:191'
        ]);


        $order_details = PaymentLogs::find($request->order_id);

        if($order_details->status == 'trial'){
            $order_details->payment_status = 'complete';
            $order_details->save();
            return redirect()->back()->with(['msg' => __('Payment Status Changed Successfully..!'),'type' => 'success']);
        }

        $order_details->status = $request->order_status;
        $order_details->payment_status = $request->order_status == 'complete' ? 'complete' : $order_details->payment_status;
        $order_details->save();

        $msg = __('Payment status changed successfully..!');

        if ($request->order_status == 'complete' && empty($order_details->tenant?->domain?->domain)) {
            LandlordPricePlanAndTenantCreate::tenant_create_event_with_credential_mail($order_details->id);
            $make_order_id = [];
            $make_order_id['order_id'] = $order_details->id;
            LandlordPricePlanAndTenantCreate::update_tenant($make_order_id);
            $msg .= ' '.__('And a new tenant has been created for the payment log');

            if($order_details->package_gateway == 'manual_payment_' && $order_details->payment_status == 'complete'){
                LandlordPricePlanAndTenantCreate::store_payment_log_history($order_details->id);
            }

            if($order_details->package_gateway == 'bank_transfer' && $order_details->payment_status == 'complete'){
                LandlordPricePlanAndTenantCreate::store_payment_log_history($order_details->id);
            }

        }else if($request->order_status == 'cancel'){

            if($request->subscription_cancel_type == 'permanent_with_delete'){
                return $this->tenant_delete_action($order_details->tenant_id);
            }

            $tenant = Tenant::find($order_details->tenant_id);
            $expire_date_con = $request->subscription_cancel_type == 'temporary' ? $order_details->expire_date : Carbon::now()->subDay(1);

            \DB::table('tenants')->where('id', $tenant->id)->update([
                'start_date' => $order_details->start_date,
                'expire_date' => $expire_date_con,
                'user_id' => $order_details->user_id,
                'theme_slug' => $order_details->theme,
                'cancel_type' => $request->subscription_cancel_type
            ]);

            $log = PaymentLogs::find($order_details->id);
            $log->status = $request->order_status;
           // $log->expire_date = $expire_date_con;
            $log->save();

        }else{
            $tenant = Tenant::find($order_details->tenant_id);
            if (is_null($tenant)){
                return redirect()->back()->with(['msg' => __("user website subdomain not found in database"),'type' => 'danger']);
            }
            \DB::table('tenants')->where('id', $tenant->id)->update([
                'start_date' => $order_details->start_date,
                'expire_date' => $order_details->expire_date,
                'user_id' => $order_details->user_id,
                'theme_slug' => $order_details->theme,
            ]);
        }

        $data['subject'] = __('your order status has been changed');
        $data['message'] = __('hello').' '.$order_details->name .'<br>';
        $data['message'] .= __('your order').' #'.$order_details->id.' ';
        $data['message'] .= __('status has been changed to').' '.str_replace('_',' ',$request->order_status).'.';

        //send mail while order status change
        try {
            Mail::to($order_details->email)->send(new BasicMail($data['message'], $data['subject']));

        }catch(\Exception $e){

            return redirect()->back()->with(['type'=> 'danger', 'msg' => $e->getMessage()]);
        }


        return redirect()->back()->with(['msg' => $msg,'type' => 'success']);
    }

    public function payment_logs_approve($id){

        $payment_logs = PaymentLogs::find($id);
        $order_details = PricePlan::find($payment_logs->package_id);

        if($payment_logs->status == 'trial'){
            $payment_logs->payment_status = 'complete';
            $payment_logs->save();
            return redirect()->back()->with(['msg' => __('Payment Status Changed Successfully..!'),'type' => 'success']);
        }

        $package_start_date = '';
        $package_expire_date = '';

        if (!empty($order_details)) {
            if ($order_details->type == 0) { //monthly
                $package_start_date = Carbon::now()->format('d-m-Y h:i:s');
                $package_expire_date = Carbon::now()->addMonth(1)->format('d-m-Y h:i:s');

            } elseif ($order_details->type == 1) { //yearly
                $package_start_date = Carbon::now()->format('d-m-Y h:i:s');
                $package_expire_date = Carbon::now()->addYear(1)->format('d-m-Y h:i:s');
            } else { //lifetime
                $package_start_date = Carbon::now()->format('d-m-Y h:i:s');
                $package_expire_date = null;
            }
        }


        if ($package_expire_date != null) {
            $old_days_left = Carbon::now()->diff($payment_logs->expire_date);
            $left_days = 0;

            if ($old_days_left->invert == 0) {
                $left_days = $old_days_left->days;
            }

            $renew_left_days = 0;
            $renew_left_days = Carbon::parse($package_expire_date)->diffInDays();

            $sum_days = $left_days + $renew_left_days;

            $new_package_expire_date = Carbon::today()->addDays($sum_days)->format("d-m-Y h:i:s");
        } else {
            $new_package_expire_date = null;
        }


        $payment_logs->payment_status = 'complete';
        $payment_logs->status = 'complete';
        $payment_logs->start_date = $payment_logs->is_renew == 1 ?  $payment_logs->start_date : $package_start_date;
        $payment_logs->expire_date = $payment_logs->is_renew == 1 ? $new_package_expire_date : $package_expire_date;
        $payment_logs->save();
        $lang = get_user_lang();


        if ($payment_logs->payment_status == 'complete')
        {
            try {

                $all_fields = [];
                $all_attachment = [];

                LandlordPricePlanAndTenantCreate::tenant_create_event_with_credential_mail($payment_logs->id);

                \DB::table('tenants')->where('id', $payment_logs->tenant_id)->update([
                    'start_date' => $payment_logs->start_date,
                    'expire_date' =>  $payment_logs->expire_date,
                    'user_id' => $payment_logs->user_id,
                    'theme_slug' => $payment_logs->theme
                ]);

                //Storing history of logs
                    if($payment_logs->package_gateway == 'manual_payment_' && $payment_logs->payment_status == 'complete'){
                        LandlordPricePlanAndTenantCreate::store_payment_log_history($payment_logs->id);
                    }

                if($payment_logs->package_gateway == 'bank_transfer' && $payment_logs->payment_status == 'complete'){
                    LandlordPricePlanAndTenantCreate::store_payment_log_history($payment_logs->id);
                }
                //Storing history of logs

            }catch (\Exception $ex){

                $message = $ex->getMessage();

                if(str_contains($message,'Access denied')){
                    LandlordPricePlanAndTenantCreate::store_exception($payment_logs->tenant_id,'domain create failed',$message,0);
                }

                return redirect()->back()->with(FlashMsg::item_delete(__('You have no permission to create database, an issue has been created, please go to website settings and manually generate this..!')));
            }


            $admin_dynamic_mail_sub = get_static_option('subscription_order_mail_admin_'.$lang.'_subject');
            $admin_dynamic_mail_mess = get_static_option('subscription_order_mail_admin_'.$lang.'_message');

            $user_dynamic_mail_sub = get_static_option('subscription_order_mail_user_'.$lang.'_subject');
            $user_dynamic_mail_mess = get_static_option('subscription_order_mail_user_'.$lang.'_message');

            $order_mail = get_static_option('order_page_form_mail');
            $order_mail = $order_mail ? $order_mail : get_static_option('site_global_email');


            //Admin Mail
            try {
                if(!empty($admin_dynamic_mail_sub) && !empty($admin_dynamic_mail_mess)){
                    Mail::to($order_mail)->send(new BasicDynamicTemplateMail(SubscriptionEmailTemplate::SubscriptionAdminMail($payment_logs)));
                }else{
                    Mail::to($order_mail)->send(new PlaceOrder($all_fields, $all_attachment, $payment_logs, "admin", 'regular'));
                }
            }catch (\Exception $ex){

            }


            try {

                //User Mail
                if(!empty($user_dynamic_mail_sub) && !empty($user_dynamic_mail_mess)){
                    Mail::to($payment_logs->email)->send(new BasicDynamicTemplateMail(SubscriptionEmailTemplate::SubscriptionUserMail($payment_logs)));
                }else{
                    Mail::to($payment_logs->email)->send(new PlaceOrder($all_fields, $all_attachment, $payment_logs, 'user', 'regular'));
                }

            }catch (\Exception $ex){

            }

        }


        //User manual payment dynamic approval Mail
        $user_manual_payment_dynamic_mail_sub = get_static_option('subscription_order_manual_payment_approved_mail_'.$lang.'_subject');
        $user_manual_payment_dynamic_mail_mess = get_static_option('subscription_order_manual_payment_approved_mail_'.$lang.'_message');

        try {

            if(!empty($user_manual_payment_dynamic_mail_sub) && !empty($user_manual_payment_dynamic_mail_mess)){
                Mail::to($payment_logs->email)->send(new BasicDynamicTemplateMail(SubscriptionEmailTemplate::SubscriptionPaymentAcceptMail($payment_logs)));
            }else{
                $subject = __('Your order payment has been approved');
                $message = __('Your order has been approved.').' #'.$payment_logs->id;
                $message .= ' '.__('Package:').' '.$payment_logs->package_name;
                Mail::to($payment_logs->email)->send(new BasicMail($message, $subject));
            }


        }catch (\Exception $e){
            return redirect()->back()->with(FlashMsg::item_delete($e->getMessage()));
        }
        //User manual payment dynamic approval Mail


        return redirect()->back()->with(['msg' => __('Order Accept Success'),'type' => 'success']);
    }


    public function order_success_payment(){
        $all_languages = Language::all();
        return view(self::ROOT_PATH.'order-success-page')->with(['all_languages' => $all_languages]);
    }

    public function update_order_success_payment(Request $request){

        $all_language = Language::all();
        foreach ($all_language as $lang) {
            $this->validate($request, [
                'site_order_success_page_' . $lang->slug . '_title' => 'nullable',
                'site_order_success_page_' . $lang->slug . '_description' => 'nullable',
            ]);
            $title = 'site_order_success_page_' . $lang->slug . '_title';
            $description = 'site_order_success_page_' . $lang->slug . '_description';

            update_static_option($title, $request->$title);
            update_static_option($description, $request->$description);
        }
        return redirect()->back()->with(['msg' => __('PaymentLogs Success Page Update Success...'),'type' => 'success']);
    }

    public function order_cancel_payment(){
        $all_languages = Language::all();
        return view(self::ROOT_PATH.'order-cancel-page')->with(['all_languages' => $all_languages]);
    }

    public function update_order_cancel_payment(Request $request){

        $all_language = Language::all();
        foreach ($all_language as $lang) {
            $this->validate($request, [
                'site_order_cancel_page_' . $lang->slug . '_title' => 'nullable',
                'site_order_cancel_page_' . $lang->slug . '_subtitle' => 'nullable',
                'site_order_cancel_page_' . $lang->slug . '_description' => 'nullable',
            ]);
            $title = 'site_order_cancel_page_' . $lang->slug . '_title';
            $subtitle = 'site_order_cancel_page_' . $lang->slug . '_subtitle';
            $description = 'site_order_cancel_page_' . $lang->slug . '_description';

            update_static_option($title, $request->$title);
            update_static_option($subtitle, $request->$subtitle);
            update_static_option($description, $request->$description);
        }
        return redirect()->back()->with(['msg' => __('PaymentLogs Cancel Page Update Success...'),'type' => 'success']);
    }

    public function bulk_action(Request $request){
        $all = PaymentLogs::find($request->ids);
        foreach($all as $item){
            $item->delete();
        }
        return response()->json(['status' => 'ok']);
    }

    public function payment_log_bulk_action(Request $request){
        $all = PaymentLogs::find($request->ids);
        foreach($all as $item){
            $item->delete();
        }
        return response()->json(['status' => 'ok']);
    }

    public function order_report(Request  $request){

        $order_data = '';
        $query = PaymentLogs::query();
        if (!empty($request->start_date)){
            $query->whereDate('created_at','>=',$request->start_date);
        }
        if (!empty($request->end_date)){
            $query->whereDate('created_at','<=',$request->end_date);
        }
        if (!empty($request->order_status)){
            $query->where(['status' => $request->order_status ]);
        }
        if (!empty($request->payment_status)){
            $query->where(['payment_status' => $request->payment_status ]);
        }
        $error_msg = __('select start & end date to generate order report');
        if (!empty($request->start_date) && !empty($request->end_date)){
            $query->orderBy('id','DESC');
            $order_data =  $query->paginate($request->items);
            $error_msg = '';
        }

        return view(self::ROOT_PATH.'order-report')->with([
            'order_data' => $order_data,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'items' => $request->items,
            'order_status' => $request->order_status,
            'payment_status' => $request->payment_status,
            'error_msg' => $error_msg
        ]);
    }

    public function payment_report(Request  $request){

        $order_data = '';
        $query = PaymentLogs::query();
        if (!empty($request->start_date)){
            $query->whereDate('created_at','>=',$request->start_date);
        }
        if (!empty($request->end_date)){
            $query->whereDate('created_at','<=',$request->end_date);
        }
        if (!empty($request->payment_status)){
            $query->where(['status' => $request->payment_status ]);
        }
        $error_msg = __('select start & end date to generate payment report');
        if (!empty($request->start_date) && !empty($request->end_date)){
            $query->orderBy('id','DESC');
            $order_data =  $query->paginate($request->items);
            $error_msg = '';
        }

        return view('landlord.admin.payment-logs.payment-report')->with([
            'order_data' => $order_data,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'items' => $request->items,
            'payment_status' => $request->payment_status,
            'error_msg' => $error_msg
        ]);
    }

    public function index(){
        $all_custom_form = FormBuilder::all();
        return view(self::ROOT_PATH.'form-section')->with(['all_custom_form' => $all_custom_form]);
    }
    public function udpate(Request $request){
        $this->validate($request,[
            'order_form' => 'nullable|string',
        ]);

        $all_language = Language::all();

        foreach ($all_language as $lang){

            $this->validate($request,[
                'order_page_'.$lang->slug.'_form_title' => 'nullable|string',
            ]);
            $field = 'order_page_'.$lang->slug.'_form_title';
            update_static_option('order_page_'.$lang->slug.'_form_title',$request->$field);
        }

        update_static_option('order_form',$request->order_form);

        return redirect()->back()->with(['msg' => __('Settings Updated....'),'type' => 'success']);
    }

    public function generate_package_invoice(Request $request)
    {
        $payment_details = PaymentLogs::find($request->id);
        if (empty($payment_details)) {
            return abort(404);
        }

        $pdf = PDF::loadview('landlord.frontend.invoice.package-order', ['payment_details' => $payment_details]) ->setOptions(['defaultFont' => 'sans-serif','isRemoteEnabled'=>true]);;

        $pdf->setPaper('L');
        $pdf->output();
        $canvas = $pdf->getDomPDF()->getCanvas();

        $height = $canvas->get_height();
        $width = $canvas->get_width();

        $canvas->set_opacity(.2,"Multiply");
        $canvas->set_opacity(.2);
        $canvas->page_text($width/5, $height/2, __('Paid'), null, 55, array(0,0,0),2,2,-30);

        return $pdf->download('package-invoice.pdf');
    }


    private function tenant_delete_action($tenant_id)
    {
        $tenant = Tenant::findOrFail($tenant_id);
        $user_id = $tenant->user_id;

        $tenant_custom_font_path = 'assets/tenant/frontend/custom-fonts/'.$tenant->id.'/';
        if(\File::exists($tenant_custom_font_path) && is_dir($tenant_custom_font_path)){
            File::deleteDirectory($tenant_custom_font_path);
        }


        //dynamic assets delete
        if(file_exists('assets/tenant/frontend/themes/css/dynamic-styles/'.$tenant->id.'-style.css')){
            unlink('assets/tenant/frontend/themes/css/dynamic-styles/'.$tenant->id.'-style.css');

        }

        if(file_exists('assets/tenant/frontend/themes/js/dynamic-scripts/'.$tenant->id.'-script.js')){
            unlink('assets/tenant/frontend/themes/js/dynamic-scripts/'.$tenant->id.'-script.js');
        }

        $path = 'assets/tenant/uploads/media-uploader/'.$tenant->id;
        CustomDomain::where([['old_domain', $tenant->id], ['custom_domain_status', '!=','connected']])
            ->orWhere([['custom_domain', $tenant->id], ['custom_domain_status', '==', 'connected']])->delete();

        PaymentLogs::where('tenant_id',$tenant->id)->delete();
        PaymentLogHistory::where('tenant_id',$tenant->id)->delete();
        PackageHistory::where('user_id',$user_id)->delete();


        if(!empty($tenant)){

            try{
                $tenant->domains()->delete();
                $tenant->delete();
            }catch(\Exception $ex){

                $message = $ex->getMessage();

                if(str_contains($message,'Access denied')){
                    return response()->danger(ResponseMessage::delete('Make sure your database user has permission to delete domain & database'));
                }

                if(str_contains($message,'drop database')){
                    return response()->danger(ResponseMessage::delete('Data deleted'));
                }
            }


        }
        if(\File::exists($path) && is_dir($path)){
            File::deleteDirectory($path);
        }



        $check_tenant = Tenant::where('user_id', $user_id)->count();
        if ($check_tenant >! 0)
        {
            User::findOrFail($user_id)->update(['has_subdomain' => false]);
        }

        return response()->danger(ResponseMessage::delete(__('Tenant deleted successfully..!')));
    }

}
